#!/bin/bash
###############################################################################
# ServerView RAID Alive ChecK script V01L00 (2012/09/27)
# Red Hat Enterprise Linux
# Copyright (c) 2012 Fujitsu Limited.
###############################################################################

VAL_CMD="$0"
VAL_OPT="$1"

LANG=C
export LANG
umask 022

###############################################################################
###############################################################################
### 
### Configuration
### 
### script schedule time by cron - [00-59] min / every hour

VALUE_RUN_MINUTE=55

### 
### 
###############################################################################
###############################################################################

### FILE/DIR DEFINITION 
CMD_BASENAME=`/bin/basename ${VAL_CMD} 2>/dev/null`
CMD_DIRNAME=`/usr/bin/dirname ${VAL_CMD} 2>/dev/null`
CMD_OBJNAME=`/bin/echo ${CMD_BASENAME} | /bin/sed -e 's/\(.*\)\.sh/\1/'`

WORKDIR=/opt/fujitsu/ServerViewSuite/RAIDManager/svrack/
SCRIPT_LOG_FILE="${WORKDIR}/trace.log"
INVALID_FLAG_FILE="/tmp/flag_svrack_invalid"

# LOCKFILE="/var/lock/subsys/aurad"
LOCKAURAD_FILE="/var/run/aurad.pid"

### CMD DEFINITION
LOGGER_TAG="ServerView RAID"
LOGGER_INFO_MSG="[109][Error] The daemon does not run properly."

AMCLI_WAIT=3
AMCLI_MAXTIMES=10
AMCLI_STDOUT_FILE=amcli_stdout
AMCLI_RETCODE_FILE=amcli_ret
AMCLI_CMD="/usr/sbin/amCLI"
AMCLI_OPTION="-l"
AMCLI_PID=""
AMCLI_RETRY_MAX=3

AURAD_CMD="/etc/init.d/aurad"
AURAD_WAIT=6
AURAD_MAXTIMES=10

DAEMON_NAME="/usr/sbin/amDaemon"

VAL_TRUE=1
VAL_FALSE=0

CRON_FILE=/var/log/cron
UPTIME=/proc/uptime
FLAG_DAEMON_RESTART=${VAL_FALSE}
RET_check_amCLI=0

/bin/mkdir -p ${WORKDIR} >/dev/null 2>&1

###############################################################################
### Trace
###############################################################################

### ATTENTION: disable (comment out) trace for release

>${SCRIPT_LOG_FILE}
/bin/env >> ${SCRIPT_LOG_FILE}

trace() {
	/bin/echo "$*" >> ${SCRIPT_LOG_FILE}
}

###############################################################################
### Success
###############################################################################
exit_with_success() {
	exit 0
}
trace define exit_with_success

###############################################################################
### Error/Hungup phenomenon occurred 
###############################################################################
exit_with_hungup() {
	exit 1
}
trace define exit_with_hungup

###############################################################################
### amCLI alive check
###############################################################################
check_amCLI() {
	### initialize
	retry=${AMCLI_RETRY_MAX}
	/bin/mv -f ${WORKDIR}/${AMCLI_RETCODE_FILE} ${WORKDIR}/${AMCLI_RETCODE_FILE}.bak >/dev/null 2>&1
	/bin/mv -f ${WORKDIR}/${AMCLI_STDOUT_FILE}  ${WORKDIR}/${AMCLI_STDOUT_FILE}.bak >/dev/null 2>&1
	
	### Retry loop 
	while [ ${retry} -gt 0 ] ; do
		times=0
		
		### amCLI background execution
		STR_DATE=`/bin/date '+%y/%m/%d %H:%M:%S' 2>/dev/null`
		( ${AMCLI_CMD} ${AMCLI_OPTION} >${WORKDIR}/${AMCLI_STDOUT_FILE}  2>&1; /bin/echo "$?" >${WORKDIR}/${AMCLI_RETCODE_FILE} ) >/dev/null 2>&1 &
		PID=$!
		/bin/sleep 1
		trace check_amCLI "amCLI: Start: ${STR_DATE}, ${AMCLI_CMD} ${AMCLI_OPTION} > ${WORKDIR}/${AMCLI_STDOUT_FILE}, ${WORKDIR}/${AMCLI_RETCODE_FILE}"
		
		### amCLI alive check loop 
		while [ ${times} -le ${AMCLI_MAXTIMES} ] ; do
			
			/bin/ps h ${PID} >/dev/null 2>&1
			RET=$?
			if [ $RET -ne 0 ] ; then
			
				### amCLI finished ### return/stdout check
				if [ -e ${WORKDIR}/${AMCLI_RETCODE_FILE} ] ; then
					STR_RET="`/bin/cat ${WORKDIR}/${AMCLI_RETCODE_FILE} 2>/dev/null`"
					STR_DATE=`/bin/date '+%y/%m/%d %H:%M:%S' 2>/dev/null`
					if [ "_${STR_RET}" = "_0" ] ; then
						trace check_amCLI "amCLI: Done: ${STR_DATE} *** SUCCESS *** ${STR_RET}"
						return ${VAL_TRUE}
					else
						STR_STDOUT="`/bin/cat ${WORKDIR}/${AMCLI_STDOUT_FILE} 2>&1`"
						trace check_amCLI "amCLI: Done: ${STR_DATE} *** ${STR_RET}, ${STR_STDOUT}"
						if [ \( "_${STR_RET}" = "_8" \) -a \( "_${STR_STDOUT}" = "_Communication with Core Service failed." \) ] ; then
							return ${VAL_FALSE}
						elif  [ \( "_${STR_RET}" = "_11" \) -a \( "_${STR_STDOUT}" = "_Database failed." \) ] ; then
							return ${VAL_FALSE}
						elif  [ \( "_${STR_RET}" = "_20" \) -a \( "_${STR_STDOUT}" = "_Database not yet initialized." \) ] ; then
							return ${VAL_TRUE}
						fi
						return ${VAL_TRUE}
					fi
				fi
			fi
			/bin/sleep ${AMCLI_WAIT}s
			times=`/usr/bin/expr ${times} + 1 2>/dev/null`
		done
	
		### TIMEOUT( amCLI cmd timeout)
		/bin/ps h ${PID} >/dev/null 2>&1
		RET=$?
		if [ $RET -eq 0 ] ; then
			/usr/bin/pkill -KILL "amCLI" >/dev/null 2>&1
		fi
		retry=`/usr/bin/expr ${retry} - 1 2>/dev/null`
		STR_DATE=`/bin/date '+%y/%m/%d %H:%M:%S' 2>/dev/null`
		trace check_amCLI "amCLI: Failed: ${STR_DATE},retry=${retry} ${PID}"
	done

	return ${VAL_FALSE}
}
trace define check_amCLI


###############################################################################
### Deamon stop / start
###############################################################################
daemon_stop(){
	### stop daemon (/etc/init.d/aurad stop)
	${AURAD_CMD} stop >/dev/null 2>&1 &

	times=0
	PID=`/bin/ps aux | /bin/grep "${DAEMON_NAME}" | /bin/grep -v grep | /bin/sed -n '1p' | /bin/awk '{print $2}'`
	if [ _ != "_${PID}" ] ; then
		trace daemon_stop "_${PID} ${times} ${AURAD_MAXTIMES}"
		while [ ${times} -le ${AURAD_MAXTIMES} ]; do
			/bin/sleep ${AURAD_WAIT}s
			/bin/ps h ${PID} >/dev/null 2>&1
			RET=$?
			if [ $RET -ne 0 ] ; then
				return
			fi
			times=`/usr/bin/expr ${times} + 1 2>/dev/null`
		done
		/bin/kill -KILL ${PID} >/dev/null 2>&1
		/bin/sleep 1
	fi
}
trace define daemon_stop

daemon_start(){
	### start daemon (/etc/init.d/aurad start)
	${AURAD_CMD} start >/dev/null 2>&1 &
	return 0
}
trace define daemon_start


###############################################################################
### syslog functions
###############################################################################
output_logger(){
  /usr/bin/logger -t "${LOGGER_TAG}" -p "$1" "$2"
}
trace define output_logger


###############################################################################
### Subroutines for Main routine
###############################################################################

### whether same process has started already.  (prevent multiple processes starting)
is_another_process_running(){
	trace ps-command "`/bin/ps -ef | /bin/grep -v grep | /bin/grep -v cron | /bin/grep ${CMD_OBJNAME}`"
	LINE=`/bin/ps -ef | /bin/grep -v grep 2>/dev/null | /bin/grep -v cron | /bin/grep -c ${CMD_OBJNAME} 2>/dev/null`
	trace is_another_process_running "${LINE}"
	if [ ${LINE} -ge 3 ] ; then
		return ${VAL_TRUE}
	fi
	return ${VAL_FALSE}
}
trace define is_another_process_running

### whether there is script invalid flag.
is_invalid_script(){
	if [ -f ${INVALID_FLAG_FILE} ] ; then
		return ${VAL_TRUE}
	fi
	return ${VAL_FALSE}
}
trace define is_invalid_script

### whether daemon running by /etc/init.d/aurad.
is_Daemon_finished_already() {
	if [ ! -e ${LOCKAURAD_FILE} ]
	then
		return ${VAL_TRUE}
	fi
	return ${VAL_FALSE}
}
trace define is_Daemon_finished_already

### whether 10 minutes have passed after OS boot. (passed=true, yet=false)
is_just_os_started(){
	VAL_UPTIME=`/bin/cat /proc/uptime 2>/dev/null | /bin/awk '{printf("%d",$1)}' 2>/dev/null`
	
	trace is_just_os_started "${VAL_UPTIME}"
	if [ ${VAL_UPTIME} -le 600 ] ; then
		return ${VAL_TRUE}
	fi
	return ${VAL_FALSE}
}
trace define is_just_os_started

### condition check at starting script
check_condition(){
	is_another_process_running
	if [ $? -eq ${VAL_TRUE} ] ; then
		trace check_condition is_another_process_running
		return ${VAL_TRUE}
	fi

	is_invalid_script
	if [ $? -eq ${VAL_TRUE} ] ; then
		trace check_condition is_invalid_script
		return ${VAL_TRUE}
	fi

	is_Daemon_finished_already
	if [ $? -eq ${VAL_TRUE} ] ; then
		# When amDaemon has finished normally by "/etc/init.d/aurad stop" already,
		trace check_condition is_Daemon_finished_already
		return ${VAL_TRUE}
	fi

	is_just_os_started
	if [ $? -eq ${VAL_TRUE} ] ; then
		trace check_condition is_just_os_started
		return ${VAL_TRUE}
	fi
	return ${VAL_FALSE}
}
trace define check_condition


error_subroutine(){
	trace error_subroutine ${FLAG_DAEMON_RESTART}
	
	if [ ${FLAG_DAEMON_RESTART} -eq ${VAL_TRUE} ] ; then
		#################################################
		### Daemon restart
		daemon_stop
		daemon_start
	fi
}


###############################################################################
### Installation mode
###############################################################################
installation(){

  /bin/cp -f ${VAL_CMD} ${WORKDIR}/${CMD_BASENAME} >/dev/null 2>&1
  trace installation file copied ${VAL_CMD} ${WORKDIR}/${CMD_BASENAME}
  
  ##################################
  # Add Schedule into Crontab
  ##################################
  TmpRoot=${WORKDIR}/workfile
  /usr/bin/crontab -l -u root >$TmpRoot 2>/dev/null
  /bin/fgrep "svrack.sh"  $TmpRoot>/dev/null || {
    /bin/echo "${VALUE_RUN_MINUTE:=30} * * * * /bin/sh -c \"[ -x ${WORKDIR}${CMD_BASENAME} ] && ${WORKDIR}${CMD_BASENAME} -c >/dev/null 2>&1 #cron\"" >> $TmpRoot
    /usr/bin/crontab -u root $TmpRoot >/dev/null 2>&1
    trace installation schedule is added into crontab
  }
  /bin/rm -f $TmpRoot
}


###############################################################################
### Uninstallation mode (only remove schedule from crontab)
###############################################################################
uninstallation(){

  ##################################
  # remove Schedule from Crontab
  ##################################
  TmpRoot=${WORKDIR}/workfile
  /usr/bin/crontab -l -u root >$TmpRoot 2>/dev/null
  /bin/fgrep "svrack.sh"  $TmpRoot >/dev/null && {
    /usr/bin/crontab -l -u root 2>/dev/null | /bin/fgrep -v "svrack.sh" >$TmpRoot 2>/dev/null
    /usr/bin/crontab -u root $TmpRoot >/dev/null 2>&1
    trace uninstallation schedule is removed from crontab
  }
  /bin/rm -f $TmpRoot
  cd /root
  /bin/rm -rf ${WORKDIR} > /dev/null 2>&1
}



###############################################################################
### Main routine
###############################################################################
trace '***START***' main routine - option="${VAL_OPT} date=`/bin/date`"

if [ "_${VAL_OPT}" = "_-i" ] ; then
  installation
  exit 0
fi

if [ "_${VAL_OPT}" = "_-r" ] ; then
  uninstallation
  exit 0
fi

if [ "_${VAL_OPT}" != "_-c" ] ; then
  /bin/echo ""
  /bin/echo "Usage: ${CMD_BASENAME} OPTION"
  /bin/echo ""
  /bin/echo "  Options:"
  /bin/echo "  -i: install to ${WORKDIR} and register the schedule into crontab"
  /bin/echo "  -c: check amDaemon status"
  /bin/echo "  -r: remove the schedule from crontab"
  /bin/echo ""
  /bin/echo "  To change schedule, please edit the value of line \"VALUE_RUN_MINUTE=xx\" in \"${WORKDIR}\" file before installation."
  
  exit 0
fi


#################################################
### status check
check_condition
if [ $? -eq ${VAL_TRUE} ] ; then
	trace check_condition
	exit_with_success
fi


#################################################
### amCLI check
check_amCLI
RET_check_amCLI=$?

if [ ${RET_check_amCLI} -ne ${VAL_TRUE} ] ; then
	FLAG_DAEMON_RESTART=${VAL_TRUE}
	trace FLAG_DAEMON_RESTART
fi


#################################################
# Judgement
trace Judgement ${RET_check_amCLI}

if [ ${RET_check_amCLI} -eq ${VAL_TRUE} ] ; then
	trace Judgement exit
	exit_with_success
fi


#################################################
# Backup investigation files and Restart daemon
#################################################

#################################################
### check amCLI error
if [ ${RET_check_amCLI} -ne ${VAL_TRUE} ] ; then
	
	#################################################
	### syslog
	trace output_logger info
	output_logger "user.info" "${LOGGER_INFO_MSG}"
fi

error_subroutine
trace error_subroutine

exit_with_hungup 1
